/*

  Copyright(c) 1999 - 2016 Intel Corporation. All rights reserved.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*
 *
 * Module Name:
 *   naltypes.h
 *
 * Abstract:
 *   NAL types for each OS are declared here.
 *
 */
#ifndef _NALTYPES_H_
#define _NALTYPES_H_

/**********************************************************************
* EFI
**********************************************************************/
#if defined(NAL_EFI)

/* Many of the NAL types are already defined in efi.h.  Types not
 * listed are the same as predefined EFI types (i.e., no point
 * in #define VOID VOID).
 *
 * EFI has own internal ASSERT macro (efidebug.h).
 */
#include <efi.h>

typedef VOID                    KVOID;
typedef UINT16                  PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;
typedef char                    CHAR;
typedef CHAR16                  WCHAR;
typedef UINT32                  NAL_STATUS;

typedef volatile UINT32         NAL_SPIN_LOCK;

#define NAL_MAX_INTERRUPT_VECTORS 256 /* Number of IRS vectors */
#define NAL_MAX_REFERENCES        32  /* Max number of ISR's per vector */

typedef struct _NAL_VECTOR
{
    volatile BOOLEAN    NalInterruptVector;
    UINT32              ReferenceCount;
} NAL_VECTOR;

/* NAL_SERVICE_ROUTINE for _NalRegisterIsr */
typedef VOID (*NAL_SERVICE_ROUTINE) (VOID);

#if defined(NAL_DEBUG)
#define ASSERTEX(X,f,l)
#else
#define ASSERTEX(X,f,l)
#endif

/**********************************************************************
* EFI2
**********************************************************************/
#elif defined(NAL_EFI2)

#include <Uefi.h>
#if defined(EFI232)
#include <Include/Ia32/ProcessorBind.h>
#elif defined(EFI264)
#include <Include/Ipf/ProcessorBind.h>
#elif defined(EFI264E)
#include <Include/X64/ProcessorBind.h>
#endif
#include <Include/sys/EfiCdefs.h>

typedef VOID                    KVOID;
typedef UINT16                  PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;
typedef char                    CHAR;
typedef CHAR16                  WCHAR;
typedef UINT32                  NAL_STATUS;

typedef volatile UINT32         NAL_SPIN_LOCK;

#define NAL_MAX_INTERRUPT_VECTORS 256 /* Number of IRS vectors */
#define NAL_MAX_REFERENCES        32  /* Max number of ISR's per vector */

typedef struct _NAL_VECTOR
{
    volatile BOOLEAN    NalInterruptVector;
    UINT32              ReferenceCount;
} NAL_VECTOR;

/* NAL_SERVICE_ROUTINE for _NalRegisterIsr */
typedef VOID (*NAL_SERVICE_ROUTINE) (VOID);

#if defined(NAL_DEBUG)
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#else
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#endif

/**********************************************************************
* DOS
**********************************************************************/
#elif defined(NAL_DOS)
/* Prevent redefinition of any of this types by Open Watcom 1.9 */
#define _EXACT_WIDTH_INTS

typedef void                    VOID;
typedef void                    KVOID;
typedef signed char             INT8;
typedef unsigned char           UINT8;
typedef signed short            INT16;
typedef unsigned short          UINT16;
typedef signed long             INT32;
typedef unsigned long           UINT32;
typedef signed __int64          INT64;
typedef unsigned __int64        UINT64;
typedef signed int              INTN;
typedef unsigned int            UINTN;

typedef UINT8                   BOOLEAN;
typedef UINT16                  PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;
typedef char                    CHAR;
typedef UINT16                  WCHAR;
typedef UINT32                  NAL_STATUS;

typedef volatile UINT32         NAL_SPIN_LOCK;

#define INLINE                  __inline

#define NAL_MAX_INTERRUPT_VECTORS 256 /* Number of IRS vectors */
#define NAL_MAX_REFERENCES        32  /* Max number of ISR's per vector */

typedef struct _NAL_VECTOR
{
    volatile BOOLEAN    NalInterruptVector;
    UINT32              ReferenceCount;
} NAL_VECTOR;

/* NAL_SERVICE_ROUTINE for _NalRegisterIsr */
typedef VOID (*NAL_SERVICE_ROUTINE) (VOID);

#if defined(NAL_DEBUG)
#include <stdio.h>
#include <stdlib.h>
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#else
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#endif

/**********************************************************************
* LINUX
**********************************************************************/
#elif defined(NAL_LINUX)

#ifdef NAL_DRIVER
/* Include function definition needed by Xen */
#include <linux/time.h>
/* Some linux flavours define VOID inside time.h */
#undef VOID
#endif

#ifndef VOID
#define VOID                    void
#endif
#define KVOID                   void
typedef signed char             INT8;
typedef unsigned char           UINT8;
typedef signed short            INT16;
typedef unsigned short          UINT16;
typedef signed int              INT32;
typedef unsigned int            UINT32;
typedef signed long long        INT64;
typedef unsigned long long      UINT64;
typedef signed long             INTN;
typedef unsigned long           UINTN;

typedef UINT8                   BOOLEAN;
typedef UINT16                  PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;
typedef char                    CHAR;
typedef UINT16                  WCHAR;
typedef UINT32                  NAL_STATUS;

#define INLINE                  static inline

#if defined(LINUXPPC64)
#define NAL_BIG_ENDIAN 1
#endif
#if defined(LINUXPPC)
#define NAL_BIG_ENDIAN 1
#endif

#if defined(NAL_DRIVER)
typedef UINTN NAL_SPIN_LOCK;
#else
#include <sys/types.h>

typedef pthread_mutex_t NAL_SPIN_LOCK;

/* These types are required to use the shared code header files. */
#include <stdint.h>
#endif

#if defined(NAL_DRIVER)
  #if defined(LINUXPPC)
    #define NAL_BIG_ENDIAN 1
  #endif
  #if defined(LINUXPPC64)
    #define NAL_BIG_ENDIAN 1
  #endif
#endif

#if defined(NAL_DEBUG)
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#else
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#endif

/**********************************************************************
* ESXi
**********************************************************************/
#elif defined(NAL_ESXI)

#ifdef NAL_DRIVER

#include <linux/time.h>

#undef VOID
#endif

#ifndef VOID
#define VOID void
#endif
#define KVOID void
typedef signed char INT8;
typedef unsigned char UINT8;
typedef signed short INT16;
typedef unsigned short UINT16;
typedef signed int INT32;
typedef unsigned int UINT32;
typedef signed long long INT64;
typedef unsigned long long UINT64;
typedef signed long INTN;
typedef unsigned long UINTN;

typedef UINT8 BOOLEAN;
typedef UINT16 PORT_ADDR;
typedef UINT64 NAL_PHYSICAL_ADDRESS;
typedef char CHAR;
typedef UINT16 WCHAR;
typedef UINT32 NAL_STATUS;

#define INLINE static inline

#if defined(NAL_DRIVER)
typedef UINTN NAL_SPIN_LOCK;
#else
#include <sys/types.h>

typedef pthread_mutex_t NAL_SPIN_LOCK;

#include <stdint.h>
#endif

#if defined(NAL_DEBUG)
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#else
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#endif

/**********************************************************************
* SOLARIS
**********************************************************************/
#elif defined(NAL_SOLARIS)

#ifndef VOID
typedef void                    VOID;
#endif
typedef void                    KVOID;
typedef signed char             INT8;
typedef unsigned char           UINT8;
typedef signed short            INT16;
typedef unsigned short          UINT16;
typedef signed int              INT32;
typedef unsigned int            UINT32;
typedef signed long long        INT64;
typedef unsigned long long      UINT64;
typedef signed long             INTN;
typedef unsigned long           UINTN;

typedef UINT8                   BOOLEAN;
typedef UINT16                  PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;
typedef char                    CHAR;
typedef UINT16                  WCHAR;
typedef UINT32                  NAL_STATUS;

#define INLINE                  static inline

#ifndef NAL_BIG_ENDIAN
#if defined(SPARC32)
#define NAL_BIG_ENDIAN 1
#endif
#if defined(SPARC64)
#define NAL_BIG_ENDIAN 1
#endif
#endif

#include <pthread.h>
typedef pthread_mutex_t NAL_SPIN_LOCK;

#if defined(NAL_DEBUG)
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#else
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#endif

#elif defined(NAL_WINNT_WDM) || defined(NAL_WINNT4)

#if defined(_WIN64)
    typedef __int64                 INTN;
    typedef unsigned __int64        UINTN;
#else
    typedef long                    INTN;
    typedef unsigned long           UINTN;
#endif

#if defined(NAL_DRIVER)
#include <ntddk.h>

#if defined(NAL_WINNT4)
    typedef unsigned char           UINT8;
    typedef signed char             INT8;
    typedef char                    CHAR;
    typedef signed short            INT16;
    typedef unsigned short          UINT16;
    typedef ULONG                   UINT32;
    typedef signed int              INT32;
    typedef ULONGLONG               UINT64;
    typedef LONGLONG                INT64;
#endif
#else
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifdef SWIG
#   define INLINE                  static inline
#endif

#include <windows.h>
#include <winioctl.h>
#endif

typedef signed char             INT8;
typedef unsigned char           UINT8;
typedef signed short            INT16;
typedef unsigned short          UINT16;
typedef void                    KVOID;
typedef UINTN                   PORT_ADDR;
typedef UINT64                  NAL_PHYSICAL_ADDRESS;

/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_STATUS
**
** \details         This type indicates a status code from a NAL function implementation should define this
**                  type a 32-bit unsigned intiger (UINT32).
**
*************************************************************************************************************/
typedef UINT32 NAL_STATUS;

typedef struct _NAL_SPIN_LOCK
{
    volatile UINT32  SpinLock;         /**< Spin Lock */
    BOOLEAN          Initialized;      /**< Initialized */
    BOOLEAN          Acquired;         /**< Acquired */
} NAL_SPIN_LOCK;

#if defined(NAL_DEBUG)
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#else
#ifndef ASSERT
#define ASSERT(X)
#endif
#ifndef ASSERTEX
#define ASSERTEX(X,f,l)
#endif
#endif

/**********************************************************************
* FreeBSD
**********************************************************************/
#elif defined(NAL_FREEBSD)

#ifndef NAL_DRIVER
#include <stdio.h>
#endif

#include <sys/types.h>

#ifndef VOID
/* user mode standard void type */
typedef void                    VOID;
#endif
/* kernel void - indicates pointer is to be used in kernel only */
typedef void                    KVOID;

/* signed 8 bit integer */
typedef signed char             INT8;
/* unsigned 8 bit integer */
typedef unsigned char           UINT8;
/* signed 16 bit integer */
typedef signed short            INT16;
/* unsigned 16 bit integer */
typedef unsigned short          UINT16;
/* signed 32 bit integer */
typedef signed int              INT32;
/* unsigned 32 bit integer */
typedef unsigned int            UINT32;
/* signed 64 bit integer */
typedef signed long long        INT64;
/* unsigned 64 bit integer */
typedef unsigned long long      UINT64;

#ifdef FREEBSD64
/* natural sized number for the platform */
typedef INT64                   INTN;
/* unsigned sized number for the platform */
typedef UINT64                  UINTN;
#else
/* natural sized number for the platform */
typedef INT32                   INTN;
/* unsigned sized number for the platform */
typedef UINT32                  UINTN;
#endif

/* should work with TRUE and FALSE */
typedef UINT8                   BOOLEAN;

/* 16 bit number for port addresses for the platform */
typedef UINT16                  PORT_ADDR;

/* 64 bit number to represent physical addresses */
typedef UINT64                  NAL_PHYSICAL_ADDRESS;

/* should be able to work as char on the platform */
typedef char                    CHAR;

/* Probably a UINT32. Will be used for status code */
typedef UINT32                  NAL_STATUS;

#ifdef NAL_DRIVER

    #include <sys/param.h>
    #include <sys/lock.h>
    #include <sys/mutex.h>

    /* structure required for SpinLocks. */
    typedef struct mtx NAL_SPIN_LOCK;

#else

    #include <pthread.h>
    /* structure required for SpinLocks. */
    typedef pthread_mutex_t NAL_SPIN_LOCK;

#endif

#if defined(NAL_DEBUG)
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#else
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#endif

#if defined(INLINE)
#undef INLINE
#endif
#define INLINE static inline
#elif defined(NAL_ESXIN)

#ifndef VOID
#   define VOID void
#endif

#include <stdint.h>

typedef uint8_t   UINT8;
typedef uint16_t  UINT16;
typedef uint32_t  UINT32;
typedef uint64_t  UINT64;
typedef  int8_t    INT8;
typedef  int16_t   INT16;
typedef  int32_t   INT32;
typedef  int64_t   INT64;

typedef UINT32 NAL_STATUS;

#ifdef ESXIN64e
typedef UINT64 UINTN;
typedef  INT64  INTN;
#else
typedef UINT32 UINTN;
typedef  INT32  INTN;
#endif

typedef UINT8  BOOLEAN;
typedef UINT16 PORT_ADDR;
typedef UINT64 NAL_PHYSICAL_ADDRESS;
typedef char   CHAR;
typedef void   KVOID;

#if defined(NAL_DRIVER)
typedef UINTN NAL_SPIN_LOCK;
#else
#include <sys/types.h>

typedef pthread_mutex_t NAL_SPIN_LOCK;

#include <stdint.h>
#endif
#define INLINE static inline

#if defined(NAL_DEBUG)
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#else
#define ASSERT(X)
#define ASSERTEX(X,f,l)
#endif

#else
/* Compilation should stop here as no OS is defined */
#error "Undefined NAL Operating System in naltypes.h."
#endif

/**********************************************************************
* Generic types for all Operating Systems.
**********************************************************************/
#ifndef IN
#define IN
#endif

#ifndef OUT
#define OUT
#endif

#ifndef NULL
#define NULL (VOID *)0
#endif

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef OPTIONAL
#define OPTIONAL
#endif

#ifndef INLINE
#define INLINE __forceinline
#endif

/* Define a constant for a NULL physical address. */
#define PHYSICAL_NULL   (NAL_PHYSICAL_ADDRESS)0

/* Used by the NalDebugPrintSupport() API. */
/*********************************************************************************************************//**
**
** \ingroup         NAL_enums
**
** \details         NAL_DEBUG_PRINT_CAPABILITIES enumeration contains debug print capabilities mask.
**
** \see             NalDebugPrintSupport,
**
*************************************************************************************************************/
typedef enum _NAL_DEBUG_PRINT_CAPABILITIES
{
    NAL_DEBUG_PRINT_SUPPORTED = 0x00000001,     /**< Debug is supported */
    NAL_DEBUG_PRINT_ASCII     = 0x00000002,     /**< ASCII is supported */
    NAL_DEBUG_PRINT_RESERVED  = 0x00000004,     /**< Reserved */
    NAL_DEBUG_PRINT_FILEIO    = 0x00000008,     /**< File I/O is supported */
    NAL_DEBUG_PRINT_SYSLOG    = 0x00000010,     /**< Debug print syslog */
    NAL_DEBUG_PRINT_COUNT
} NAL_DEBUG_PRINT_CAPABILITIES;

/* Used by the NalAllocateMemory() APIs. */
#define NAL_SIZE_OF_8_BYTES         8

/* Used by the 32-bit ISR APIs. */
#define NAL_PIC1_FIRST_ACCESS_PORT  0x0020
#define NAL_PIC2_FIRST_ACCESS_PORT  0x00A0
#define NAL_PIC1_SECOND_ACCESS_PORT 0x0021
#define NAL_PIC2_SECOND_ACCESS_PORT 0x00A1
#define NAL_CLEAR_PIC_COMMAND       0x0020

/* Used by the timer callback APIs. */
#define NAL_TIMER_INTERVAL_LENGTH   100 /* 100 nanoseconds */
#define NAL_MAX_TIMER_CALLBACKS     1   /* Maximum number of timer callbacks allowed. */
/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_TIMER_CALLBACK
**
** \details         This is a ReceiveCallback function. See receive.c from sample app, which shows
**                  NAL_TIMER_CALLBACK usage. It is defined as:
**
** \code
** typedef VOID (*NAL_TIMER_CALLBACK) (VOID*);
** \endcode
**
*************************************************************************************************************/
typedef VOID (*NAL_TIMER_CALLBACK) (VOID*);

/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_THREAD_FUNC
**
** \details         This type is used to provide the unified pointer to the thread call function. It is defined
**                  as:
**
** \code
** typedef VOID *(*NAL_THREAD_FUNC) (VOID*);
** \endcode
**
*************************************************************************************************************/
typedef VOID *(*NAL_THREAD_FUNC) (VOID*);

#ifndef NAL_DRIVER
/* NAL_TIMER_CALLBACK for NalRegisterTimerCallback */

typedef struct _NAL_TIMER_CALLBACK_ENTRY
{
    struct _NAL_TIMER_CALLBACK_ENTRY* Next;
    struct _NAL_TIMER_CALLBACK_ENTRY* Prev;

    volatile UINT64     NextExecution;  /* UINT64 reference to some OS time to
                                         * next execute the timer callback.
                                         */

    NAL_TIMER_CALLBACK  TimerCallback;  /* Set by NalRegisterTimerCallback. */
    VOID*               Context;        /* Set by NalRegisterTimerCallback. */

    UINT32              CallbackId;     /* Set by NalRegisterTimerCallback. */

    UINT32              TimerInterval;  /* Set by NalRegisterTimerCallback. Used
                                         * to determine if the callback should be
                                         * executed again.
                                         */

    UINT32              Status;         /* This is the callback status and will
                                         * be set to NONE, WAITING, RESCHEDULE,
                                         * EXECUTING, or REMOVE.
                                         */

} NAL_TIMER_CALLBACK_ENTRY;

#define NAL_TIMER_CALLBACK_ID_INVALID           0
#define NAL_TIMER_CALLBACK_ID_NEW_ENTRY         -1

#define NAL_TIMER_CALLBACK_STATUS_NONE          0
#define NAL_TIMER_CALLBACK_STATUS_WAITING       1
#define NAL_TIMER_CALLBACK_STATUS_RESCHEDULE    2
#define NAL_TIMER_CALLBACK_STATUS_EXECUTING     3
#define NAL_TIMER_CALLBACK_STATUS_REMOVE        4

#endif
/*********************************************************************************************************//**
**
** \ingroup      NAL_enums
**
** \details      Used for determining the Alert Standard Format support of an adapter.
**
** \note            ASF term represents a wide range of technologies that enable remote system access and
**                  control in both OS-present and OS-absent environments.
**                  Alert Standard Format (ASF) provides remote notifications of client system states and
**                  hardware or software failures, without regard to operating system or system power state.
**
*************************************************************************************************************/
typedef enum _NAL_ASF_SUPPORT
{
	NAL_ASF_NOT_SUPPORTED = 0,      /**< ASF is not supported */
	NAL_ASF1_SUPPORTED,             /**< ASF1 supported */
	NAL_ASF2_SUPPORTED,             /**< ASF2 supported */
	NAL_ASF1_ASF2_SUPPORTED         /**< ASF1 and ASF2 is supported */
} NAL_ASF_SUPPORT;

/*********************************************************************************************************//**
**
** \ingroup         NAL_enums
**
** \details         Operating system detection enumeration constants. These are used with the
**                  NalIsTargetOperatingSystem function.
**
*************************************************************************************************************/
typedef enum _NAL_OS_TYPE
{
	NAL_OS_UNKNOWN = 0,         /**< Unknown */
    NAL_OS_DOS,                 /**< DOS */
    NAL_OS_EFI32,               /**< EFI 32 */
    NAL_OS_EFI64,               /**< EFI 64*/
    NAL_OS_LINUX32,             /**< LINUX 32 */
    NAL_OS_WIN3XX,              /**< WIN3XX */
    NAL_OS_WIN9X,               /**< WIN9X */
    NAL_OS_OS2,                 /**< OS2 */
    NAL_OS_WINNT4,              /**< WINNT4 */
    NAL_OS_WIN2K,               /**< WIN2K */
    NAL_OS_WINXP32,             /**< WINPX 32 */
    NAL_OS_WINXP64,             /**< WINPX 64 */
    NAL_OS_WINXP64E,            //**< WINP 64E */
    NAL_OS_LINUX64,             /**< LINUX 64 */
    NAL_OS_FREEBSD32,           /**< FREEBSD 32 */
    NAL_OS_FREEBSD64e,          /**< FREEBSD 64e */
    NAL_OS_LINUX64E,            /**< LINUX 64e */
    NAL_OS_ESXI32,              /**< ESXI 32 */
    NAL_OS_ESXI64E,             /**< ESXI 64e */
    NAL_OS_NWS,                 /**< NWS */
    NAL_OS_EFI64E,              /**< EFI64e */
    NAL_OS_LINUXPPC,            /**< LINUXPCC */
    NAL_OS_LINUXPPC64,          /**< LINUXPCC 64 */
    NAL_OS_COUNT,               /**< COUNT */
    NAL_OS_SOLARIS_X86,         /**< SOLARIX X86 */
    NAL_OS_SOLARIS_64E,         /**< SOLARIS 64e */
    NAL_OS_SOLARIS_SPARC32,     /**< SOLARIS SPARC 32 */
    NAL_OS_SOLARIS_SPARC64      /**< SPLARIS SPARC 64 */
} NAL_OS_TYPE;

/* Operating system kernel run domain. These are returned from
 * the _NalGetOsRunDomain function.
 */
typedef enum _NAL_OS_RUN_DOMAIN
{
    NAL_OS_DOMAIN_BAREMETAL = 0,
    NAL_OS_DOMAIN_0,
    NAL_OS_DOMAIN_U
} NAL_OS_RUN_DOMAIN;

#define NAL_OS_LINUX   NAL_OS_LINUX32

/* These types are required to use the shared code header files. */
#if !defined(NAL_DRIVER)
#define boolean_t   BOOLEAN
#define uint8_t     UINT8
#define u8          UINT8
#define int8_t      INT8
#define s8          INT8
#define uint16_t    UINT16
#define u16         UINT16
#define int16_t     INT16
#define s16         INT16
#define uint32_t    UINT32
#define u32         UINT32
#define int32_t     INT32
#define s32         INT32
#define uint64_t    UINT64
#define u64         UINT64
#define int64_t     INT64
#define s64         INT64
#endif

/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_REGISTER
**
** \details         NAL_REGISTER data type is used to specify a particular MAC or PHY register. It is 32 bit
**                  value.
**
*************************************************************************************************************/
typedef UINT32  NAL_REGISTER;
/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_MAC_TYPE
**
** \details         This value is used to determinate the MAC controller type. This value should be tested
**                  against a named value, and the order of the apperiance in the enumeration, which
**                  represents version of MAC controller and the capabilities of the cotroller.
**
*************************************************************************************************************/
typedef UINTN   NAL_MAC_TYPE;

/* Macro to make 0/1 only from 0/non-0
 * For example to satisfy "secure" 2 items array indexing with bool*/
#define MAKE_BOOLEAN(x) (!!(x))

/* This enum is used to decide if we are dealing with the Rx or the Tx queue.
 * It is mainly used with the 40G adapters.
 */
typedef enum _NAL_QUEUE_TYPE
{
    NAL_RX_QUEUE = 0,
    NAL_TX_QUEUE
} NAL_QUEUE_TYPE;

typedef struct _NAL_NVM_FEATURE_CONFIG
{
    UINT16  Id;
    UINT16  Options;
    UINT16  Selection;
} NAL_NVM_FEATURE_CONFIG;

typedef enum _NAL_DRIVER_MODE
{
    NAL_DRIVER_MODE_QV_DRIVER  ,
    NAL_DRIVER_MODE_BASE_DRIVER ,
    NAL_DRIVER_MODE_NO_DRIVER ,
} NAL_DRIVER_MODE;

/*********************************************************************************************************//**
**
** \ingroup         Data_Types
**
** \typedef         NAL_OS_INIT_CONFIG
**
** \details         NAL_OS_INIT_CONFIG is structure with NAL OS initialization data required
**                  for NalInitializeOsEx
**
*************************************************************************************************************/
typedef struct
{
    BOOLEAN         ForceDriverless;
    BOOLEAN         FailoverDriverless;
    BOOLEAN         SkipQvDriverVerification;
} NAL_OS_INIT_CONFIG;

typedef enum
{
    NAL_BASE_DRIVER_ID_IGBE  ,
    NAL_BASE_DRIVER_ID_IXGBE ,
    NAL_BASE_DRIVER_ID_I40E ,
    NAL_BASE_DRIVER_ID_NUMBER_OF_BASE_DRIVERS ,
} NAL_BASE_DRIVER_ID;

#endif
